
import numpy as np

from bokeh.io import curdoc
from bokeh.layouts import row, widgetbox
from bokeh.models import ColumnDataSource
from bokeh.models.widgets import Slider, Button
from bokeh.plotting import figure

def normal_pdf(x, mu, var):
    return np.exp(-np.power(x-mu, 2)/(2*var)) / np.sqrt(2 * np.pi * var)

def compute_high_density(p, alpha, N):
    p_sort = np.sort(p)
    p_cs = np.cumsum(p_sort)
    i_val = np.searchsorted(p_cs, alpha)
    if i_val < N:
        val = p_sort[i_val]
    else:
        val = 1.0
        
    return p > val

mu1 = 0.0
mu2 = 0.0
sigma = 1.0
alpha = .05
xllim = -7.5
xulim = 7.5

# Set up data
N = 250
x = np.linspace(xllim, xulim, N)
y1 = .5*normal_pdf(x, mu1, sigma**2)
y2 = .5*normal_pdf(x, mu2, sigma**2)
N1 = ColumnDataSource(data=dict(x=x, y=y1+y2))

y_obs = np.array((0,0.2))

# Set up plot
plot = figure(plot_height=400, plot_width=800, title="Highest density region",
              tools="crosshair,pan,reset,save,wheel_zoom",
              x_range=[xllim, xulim], y_range=[0, .5])

plot.line('x', 'y', source=N1, line_width=3, line_alpha=0.6, legend='probability density function')

proba = (y1 + y2)*(xulim - xllim)/(N-1)
is_hdr = compute_high_density(proba, alpha, N)
x_circle = x[is_hdr]
y_circle = 0.005 * np.ones(x_circle.shape)
N3 = ColumnDataSource(data=dict(x=x_circle, y=y_circle))
plot.circle('x', 'y', source=N3, size=2, color="firebrick", alpha=0.5, legend='highest density region')

# Set up widgets
mu1_slider = Slider(title="mu_1", value=mu1, start=-5.0, end=5.0, step=0.1)
mu2_slider = Slider(title="mu_2", value=mu2, start=-5.0, end=5.0, step=0.1)
alpha_slider = Slider(title="alpha", value=alpha, start=0, end=1, step=.01)

def update(attrname, old, new):
    mu1 = mu1_slider.value
    mu2 = mu2_slider.value
    y1 = .5*normal_pdf(x, mu1, sigma**2)
    y2 = .5*normal_pdf(x, mu2, sigma**2)

    N1.data = dict(x=x, y=y1+y2)

    alpha = alpha_slider.value
    proba = (y1 + y2)*(xulim - xllim)/N
    is_hdr = compute_high_density(proba, alpha, N)
    x_circle = x[is_hdr]
    y_circle = 0.005 * np.ones(x_circle.shape)
    N3.data = dict(x=x_circle, y=y_circle)

mu1_slider.on_change('value', update)
mu2_slider.on_change('value', update)
alpha_slider.on_change('value', update)

# Set up layouts and add to document
inputs = widgetbox(mu1_slider, mu2_slider, alpha_slider)

curdoc().add_root(row(inputs, plot, width=800))
curdoc().title = "Highest density region"

