
# This file is a free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This file is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# Copyright (C) 2019 Jeremie Houssineau
#
# Support: jeremie.houssineau AT warwick.ac.uk
#

import numpy as np
from scipy.special import gamma

from bokeh.io import curdoc
from bokeh.layouts import row, widgetbox
from bokeh.models import ColumnDataSource
from bokeh.models.widgets import Slider, Dropdown
from bokeh.plotting import figure

# Define p.d.f.s of interest

def gamma_pdf(x, alpha, beta):
    return np.power(beta, alpha) / gamma(alpha) * np.power(x, alpha-1) * np.exp(-beta * x)

def normal_pdf(x, mu, sigma):
    return np.exp(-np.power(x-mu, 2)/(2*sigma**2)) / (np.sqrt(2 * np.pi) * sigma)

def beta_pdf(x, alpha, beta):
    return gamma(alpha+beta) / (gamma(alpha) * gamma(beta)) * np.power(x, alpha-1) * np.power(1-x, beta-1)

def student_pdf(x, nu):
    return gamma(.5*(nu+1)) / (np.sqrt(nu*np.pi) * gamma(.5*nu)) * np.power(1 + np.power(x, 2)/nu, -.5*(nu+1))

# Define drop-down menu

menu = [("Gamma p.d.f.", "gamma"), ("Normal p.d.f.", "normal"),\
    ("Beta p.d.f.", "beta"), ("Student's t p.d.f.", "student")]
dropdown = Dropdown(label="P.d.f", menu=menu) # button_type="warning"

# Number of points to plot the p.d.f.s (reduce this number of speed)
N = 500

# x and y coordinates are initialised with a Dirac function
x = np.array((-5, 0, 0, 0, 5))
y = np.array((0, 0, 99999, 0, 0))
source = ColumnDataSource(data=dict(x=x, y=y))

# Set up plot
plot = figure(plot_height=400, plot_width=800, title="", tools="crosshair,pan,reset,save,wheel_zoom",
              x_range=(-5, 5), y_range=(0, 1.1))

plot.line('x', 'y', source=source, line_width=3, line_alpha=0.6)

# Set up widgets
param1 = Slider(title="param1", value=0.5, start=0.0, end=1.0, step=0.05)
param2 = Slider(title="param2", value=0.5, start=0.0, end=1.0, step=0.05)

# Function modifying the parameters depending on the choice of p.d.f.
def update_dist(attrname, old, new):
    if dropdown.value == 'gamma':
        param1.title = 'alpha'
        param1.start = 0.05
        param1.end = 5.0
        param1.value = 1
        param2.title = 'beta'
        param2.start = 0.05
        param2.end = 5.0
        param2.value = 1
        plot.title.text = "Gamma p.d.f."
        plot.x_range.start = 0
        plot.x_range.end = 10
        plot.y_range.start = 0
        plot.y_range.end = 1.1
    
    elif dropdown.value == 'normal':
        param1.title = 'mu'
        param1.start = -2.5
        param1.end = 2.5
        param1.value = 0
        param2.title = 'sigma'
        param2.start = 0.05
        param2.end = 5.0
        param2.value = 1
        plot.title.text = "Normal p.d.f."
        plot.x_range.start = -5
        plot.x_range.end = 5
        plot.y_range.start = 0
        plot.y_range.end = 1.1

    elif dropdown.value == 'beta':
        param1.title = 'alpha'
        param1.start = 0.05
        param1.end = 5.0
        param1.value = 1
        param2.title = 'beta'
        param2.start = 0.05
        param2.end = 5.0
        param2.value = 1
        plot.title.text = "Beta p.d.f."
        plot.x_range.start = 0.01
        plot.x_range.end = 0.99
        plot.y_range.start = 0
        plot.y_range.end = 2.5

    elif dropdown.value == 'student':
        param1.title = 'nu'
        param1.start = 0.5
        param1.end = 10
        param1.value = 1
        param2.title = '--'
        param2.start = 0
        param2.end = 1
        param2.value = 0.5
        plot.title.text = "Student's t p.d.f."
        plot.x_range.start = -5
        plot.x_range.end = 5
        plot.y_range.start = 0
        plot.y_range.end = 0.5

dropdown.on_change('value', update_dist)

# Function modifying the x and y coordinates depending on the parameters
def update_data(attrname, old, new):
    if dropdown.value == 'gamma':
        x = np.linspace(0.01, 10, N)
        y = gamma_pdf(x, param1.value, param2.value)
    
    elif dropdown.value == 'normal':
        x = np.linspace(-5, 5, N)
        y = normal_pdf(x, param1.value, param2.value)

    elif dropdown.value == 'beta':
        x = np.linspace(0.01, 0.99, N)
        y = beta_pdf(x, param1.value, param2.value)

    elif dropdown.value == 'student':
        x = np.linspace(-5, 5, N)
        y = student_pdf(x, param1.value)
    
    source.data = dict(x=x, y=y)

param1.on_change('value', update_data)
param2.on_change('value', update_data)

# Set up layouts and add to document
inputs = widgetbox(dropdown, param1, param2)

curdoc().add_root(row(inputs, plot, width=800))
curdoc().title = "2-parameter p.d.f.s"

